// Quellen: 
// https://sqlsunday.com/2014/08/17/30-360-day-count-convention/
// https://strata.opengamma.io/day_counts/
// https://en.wikipedia.org/wiki/Day_count_convention
// https://www.isda.org/a/AIJEE/1998-ISDA-memo-%E2%80%9CEMU-and-Market-Conventions-Recent-Developments%E2%80%9D.pdf
//
// Die Funktion BRTEILJAHRE_30US_360 wandelt die Anzahl der ganzen Tage zwischen Ausgangsdatum und Enddatum in Bruchteile von Jahren um
// Sie entspricht der Funktion BRTEILJAHRE(Ausgangsdatum;Enddatum;Basis) in Microsoft Excel mit der Basis 0 => US (NASD) 30/360.
// Erstellt am 07.11.2021 von Frank Arendt-Theilen, Hameln
//

(Ausgangsdatum as date, Enddatum as date)  as number =>
let

    Ausgangsdatum_Monat = Date.Month(Ausgangsdatum),
    Ausgangsdatum_Jahr = Date.Year(Ausgangsdatum),
    Ausgangsdatum_AlsLetzterTagImFebruar = Date.EndOfMonth(#date(Ausgangsdatum_Jahr, 2, 1)),

    Enddatum_Monat = Date.Month(Enddatum),
    Enddatum_Jahr = Date.Year(Enddatum),
    Enddatum_AlsLetzterTagImFebruar = Date.EndOfMonth(#date(Enddatum_Jahr, 2, 1)),

    //Wenn das Ausgangsdatum und das Enddatum jeweils der letzte Tag des Februars sind, dann den Tag des Enddatums auf den 30. abändern
    Enddatum_Tag = 
        if Ausgangsdatum = Ausgangsdatum_AlsLetzterTagImFebruar and Enddatum = Enddatum_AlsLetzterTagImFebruar then 30
        else
            //Wenn der Tag des Enddatums größer ist als 30 und der Tag des Ausgangsdatum größer gleich 30 ist, dann den Tag des Enddatums auf den 30. abändern
            if Date.Day(Ausgangsdatum) >= 30 and Date.Day(Enddatum) = 31 then 30
            else Date.Day(Enddatum),
    //Wenn nur das Ausgangsdatum der letzte Tag des Februars ist, dann den Tag des Ausgangsdatums auf den 30. abändern
    Ausgangsdatum_Tag =
        if Ausgangsdatum = Ausgangsdatum_AlsLetzterTagImFebruar then 30
        else
            //Wenn der Tag des Ausgangsdatums der 31. ist, dann den Tag des Ausgangsdatums auf den 30. abändern
            if Date.Day(Ausgangsdatum) = 31 then 30
            else Date.Day(Ausgangsdatum),


    BRTEILJAHRE_30US_360 = 
        if Enddatum >= Ausgangsdatum then
            ( 360 * ( Enddatum_Jahr - Ausgangsdatum_Jahr) + 30 * ( Enddatum_Monat - Ausgangsdatum_Monat ) + ( Enddatum_Tag - Ausgangsdatum_Tag )) / 360
        else
            ( 360 * ( Ausgangsdatum_Jahr - Enddatum_Jahr) + 30 * ( Ausgangsdatum_Monat - Enddatum_Monat ) + ( Ausgangsdatum_Tag - Enddatum_Tag )) / 360

in
    BRTEILJAHRE_30US_360